'use strict';

/* --------------------------------------------------------------
 slides_container.js 2016-12-29
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Controller Module For Slides Container (Tabs)
 *
 * Handles the sliders container functionality in the sliders details page.
 */
gx.controllers.module('slides_container', [jse.source + '/vendor/jquery-deparam/jquery-deparam.min.js', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.css', jse.source + '/vendor/jquery-ui-dist/jquery-ui.min.js', 'xhr', 'modal'], function (data) {

    'use strict';

    // --------------------------------------------------------------------
    // VARIABLES
    // --------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);
    var $footer = $('.gx-admin-footer');

    /**
     * Elements
     *
     * @type {Object}
     */
    var elements = {
        // Buttons.
        buttons: {
            // Sort mode button.
            sort: $this.find('.sort-button'),

            // Create button.
            create: $this.find('.btn-create'),

            // Submit button group.
            submit: $footer.find('.submit-button-group'),

            // Submit button for save slider
            submitSave: $footer.find('.save'),

            // Submit button for save and refresh slider
            submitRefresh: $footer.find('.refresh')
        },

        // Template.
        templates: {
            // Slide panel set template.
            slidePanel: $this.find('#slide-panel-template')
        },

        // Modals.
        modals: {
            // Delete image modal.
            deleteImage: $('.delete-image.modal'),

            // Delete slide modal.
            deleteSlide: $('.delete-slide.modal'),

            // Edit image map modal.
            imageMap: $('.image-map.modal')
        },

        // Tabs.
        tabHeader: $this.find('.nav-tabs'),

        // Select box which holds all images that will be deleted.
        deleteImageSelect: $('#delete_images')
    };

    /**
     * CSS class names.
     *
     * @type {Object}
     */
    var classes = {
        // New image.
        newImage: 'new'
    };

    /**
     * Selector Strings
     *
     * @type {Object}
     */
    var selectors = {
        // Icon selector strings.
        icons: {
            // Delete button on the panel header.
            delete: '.icon.delete',

            // Drag button on the panel header.
            drag: '.drag-handle',

            // Collapser button on the panel header.
            collapser: '.collapser',

            // Image delete button.
            imageDelete: '.action-icon.delete',

            // Image map edit button.
            imageMap: '.action-icon.image-map',

            // Upload image button.
            upload: '.action-icon.upload'
        },

        // Inputs selector strings.
        inputs: {
            // General image select dropdowns.
            dropdown: '.dropdown-input',

            // Thumbnail dropdown.
            thumbnailImageDropdown: '[name="thumbnail"]',

            // Title.
            title: 'input[name="title"]',

            // File.
            file: '.file-input'
        },

        // Slide panel.
        slidePanel: '.panel',

        // Tab body.
        tabBody: '.tab-pane',

        // Slide panel title.
        slidePanelTitle: '.slide-title',

        // Setting row (form group).
        configurationRow: '.row.form-group',

        // Data list container for image map.
        imageMapDataList: '.image-map-data-list'
    };

    /**
     * Cache list of open slide panels.
     *
     * @type {jQuery[]}
     */
    var openSlidePanels = [];

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // --------------------------------------------------------------------
    // FUNCTIONS
    // --------------------------------------------------------------------

    /**
     * Registers a change, so that the user gets a confirmation dialog while leaving the page.
     */
    function _registerChange() {
        // Object of GET parameters.
        var getParameters = $.deparam(window.location.search.slice(1));

        // Only register in slider edit mode.
        if ('id' in getParameters) {
            window.onbeforeunload = function () {
                return jse.core.lang.translate('EXIT_CONFIRMATION_TEXT', 'sliders');
            };
        }
    }

    /**
     * Handles the image dropdown change event.
     *
     * @param {jQuery.Event} event Triggered event.
     * @param {Boolean} [removeAllDataListItems = false] Remove all data list container list items?
     */
    function _onImageChange(event) {
        var removeAllDataListItems = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : true;

        // Image dropdown element.
        var $dropdown = $(event.target);

        // Remove icon element.
        var $removeIcon = $dropdown.parents(selectors.configurationRow).find(selectors.icons.imageDelete);

        // Image map icon element.
        var $imageMapIcon = $dropdown.parents(selectors.configurationRow).find(selectors.icons.imageMap);

        // Image map data container list element.
        var $list = $dropdown.parents(selectors.configurationRow).find(selectors.imageMapDataList);

        // Remove the remove icon if 'do not use' is selected.
        $removeIcon[$dropdown.val() ? 'show' : 'hide']();

        // Remove the image map icon if 'do not use' is selected.
        $imageMapIcon[$dropdown.val() ? 'show' : 'hide']();

        // Empty image map data container list.
        $list.children().filter(removeAllDataListItems ? '*' : '.new').remove();
    }

    /**
     * Triggers the file select (click) event of the invisible file input field.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onUploadIconClick(event) {
        $(event.target).parents(selectors.configurationRow).find(selectors.inputs.file).trigger('click');
    }

    /**
     * Handles the file select (change) event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageAdd(event) {
        // Exit method, if selection has been aborted.
        if (!event.target.files.length) {
            return;
        }

        // File input element.
        var $fileInput = $(event.target);

        // Image dropdown.
        var $dropdown = $fileInput.parents(selectors.configurationRow).find(selectors.inputs.dropdown);

        // Regular expression to validate the file name.
        var regex = /(.)(jpg|jpeg|png|gif|bmp|webp)$/i;

        // File name.
        var fileName = event.target.files[0].name;

        // Is the dropdown for thumbnail images?
        var isThumbnailImage = !!$fileInput.parents(selectors.configurationRow).find(selectors.inputs.thumbnailImageDropdown).length;

        // Exit method and show modal, if file type does not match.
        if (!regex.test(fileName)) {
            // Show modal.
            jse.libs.modal.showMessage(jse.core.lang.translate('INVALID_FILE_MODAL_TITLE', 'sliders'), jse.core.lang.translate('INVALID_FILE_MODAL_TEXT', 'sliders'));

            // Reset value.
            $fileInput.val('');

            return;
        }

        // Exit method and show modal, if filename is already present in dropdown.
        var _iteratorNormalCompletion = true;
        var _didIteratorError = false;
        var _iteratorError = undefined;

        try {
            for (var _iterator = $dropdown[0].children[Symbol.iterator](), _step; !(_iteratorNormalCompletion = (_step = _iterator.next()).done); _iteratorNormalCompletion = true) {
                var $option = _step.value;

                // Check if option's text content matches with the name of the selected file.
                if ($option.textContent === fileName) {
                    // Show modal.
                    jse.libs.modal.showMessage(jse.core.lang.translate('FILENAME_ALREADY_USED_MODAL_TITLE', 'sliders'), jse.core.lang.translate('FILENAME_ALREADY_USED_MODAL_TEXT', 'sliders'));

                    // Reset value.
                    $fileInput.val('');

                    return;
                }
            }

            // Add files to dropdowns.
        } catch (err) {
            _didIteratorError = true;
            _iteratorError = err;
        } finally {
            try {
                if (!_iteratorNormalCompletion && _iterator.return) {
                    _iterator.return();
                }
            } finally {
                if (_didIteratorError) {
                    throw _iteratorError;
                }
            }
        }

        _addImageToDropdowns(fileName, isThumbnailImage);

        // Select value.
        $dropdown.val(fileName).trigger('change');
    }

    /**
     * Handles the image delete button click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageDelete(event) {
        // Loading CSS class name.
        var loadingClass = 'loading';

        // Image dropdown container.
        var $configurationRow = $(event.target).parents(selectors.configurationRow);

        // Image dropdown.
        var $dropdown = $configurationRow.find(selectors.inputs.dropdown);

        // Slide ID.
        var slideId = $configurationRow.parents(selectors.slidePanel).data('id');

        // Slide image ID.
        var slideImageId = $configurationRow.data('id');

        // Is the dropdown for thumbnail images?
        var isThumbnailImage = !!$dropdown.is(selectors.inputs.thumbnailImageDropdown);

        // Selected file name.
        var fileName = $dropdown.val();

        // Add loading state.
        $dropdown.addClass(loadingClass);

        // Image usage check request options.
        var requestOptions = {
            url: 'admin.php?do=SlidersDetailsAjax/CheckImageUsage',
            data: {
                filename: fileName,
                is_thumbnail: isThumbnailImage,
                slide_id: slideId,
                slide_image_id: slideImageId
            }
        };

        // Perform deletion.
        var performDeletion = function performDeletion() {
            // Put image name into deleter select box.
            elements.deleteImageSelect.append($('<option>', {
                val: fileName,
                class: isThumbnailImage ? 'thumbnail' : ''
            }));

            // Delete image from dropdowns.
            _deleteImageFromDropdowns(fileName, isThumbnailImage);
        };

        // Check image usage.
        jse.libs.xhr.get(requestOptions).then(function (response) {
            // Remove loading state.
            $dropdown.removeClass(loadingClass);

            if (response.isUsed) {
                // Modal confirmation button element.
                var $confirmButton = elements.modals.deleteImage.find('button.confirm');

                // Show modal.
                elements.modals.deleteImage.modal('show');

                // Listen to confirmation button click event.
                $confirmButton.off('click').on('click', performDeletion);
            } else {
                performDeletion();
            }
        });
    }

    /**
     * Handles the image map edit button click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onImageMap(event) {
        // Slide image ID.
        var slideImageId = $(event.target).parents(selectors.configurationRow).data('id');

        // List element which acts like a data container.
        var $list = $(event.target).parents(selectors.configurationRow).find(selectors.imageMapDataList);

        // Image dropdown.
        var $dropdown = $(event.target).parents(selectors.configurationRow).find(selectors.inputs.dropdown);

        // Slide image file name.
        var imageFilename = $dropdown.val();

        // Is a new image selected?
        var isNewImageSelected = $dropdown.find('option:selected').hasClass(classes.newImage);

        // Path to image URL.
        var imageUrl = jse.core.config.get('appUrl') + '/images/slider_images/' + imageFilename;

        // Show save first notice modal and return immediately, if the slide image has no ID.
        if (!slideImageId || isNewImageSelected) {
            jse.libs.modal.showMessage(jse.core.lang.translate('IMAGE_MAP_MODAL_TITLE', 'sliders'), jse.core.lang.translate('SAVE_SLIDER_FIRST_NOTICE_TEXT', 'sliders'));

            return;
        }

        // Show image map modal.
        elements.modals.imageMap.trigger('show', [$list, imageUrl]);
    }

    /**
     * Handles the sort button click event.
     */
    function _onSortButtonClick() {
        // Indicator CSS classes.
        var indicatorClass = 'mode-on btn-primary';

        // Selector string for the slide panel body.
        var slidePanelBodySelector = '.panel-body';

        // Slides container tabs, except the active one.
        var $otherTabs = elements.tabHeader.children().not('.active');

        // Is the sort mode on?
        var isModeOn = elements.buttons.sort.hasClass(indicatorClass);

        // Language-specific button texts.
        var enterText = elements.buttons.sort.data('textEnter');
        var exitText = elements.buttons.sort.data('textExit');

        // All slide panels.
        var $slides = $this.find(selectors.slidePanel);

        // Apply fade effect onto slide panels.
        $slides.hide().fadeIn();

        // Switch text and toggle indicator class.
        elements.buttons.sort[isModeOn ? 'removeClass' : 'addClass'](indicatorClass).text(isModeOn ? enterText : exitText);

        // Toggle create button.
        elements.buttons.create.prop('disabled', !isModeOn);

        // Toggle drag handle buttons.
        $slides.find(selectors.icons.drag)[isModeOn ? 'hide' : 'show']();

        // Toggle other tabs.
        $otherTabs[isModeOn ? 'fadeIn' : 'fadeOut']();

        // Toggle collapser and hide buttons.
        $slides.find(selectors.icons.collapser).add(selectors.icons.delete)[isModeOn ? 'show' : 'hide']();

        // Save open slide panels.
        if (!isModeOn) {
            openSlidePanels = $slides.filter(function (index, element) {
                return $(element).find(slidePanelBodySelector).is(':visible');
            });
        }

        // Toggle saved open slide panels.
        openSlidePanels.each(function (index, element) {
            return $(element).find(selectors.icons.collapser).trigger('click');
        });
    }

    /**
     * Handles the sort start event.
     */
    function _onSortStart() {
        // Tab content element for selected language.
        var $tabBody = $this.find(selectors.tabBody).filter(':visible');

        // Refresh tab sizes and positions.
        $tabBody.sortable('refreshPositions');
    }

    /**
     * Handles the sort stop event.
     */
    function _onSortStop() {
        // Register change, to make prompt on page unload.
        _registerChange();
    }

    /**
     * Handles the delete icon click event.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onDeleteIconClick(event) {
        // Slide panel element.
        var $slidePanel = $(event.target).parents(selectors.slidePanel);

        // Modal confirmation button element.
        var $confirmButton = elements.modals.deleteSlide.find('button.confirm');

        // Show modal.
        elements.modals.deleteSlide.modal('show');

        // Listen to confirmation button click event.
        $confirmButton.off('click').on('click', function () {
            return _onDeleteConfirmationButtonClick(elements.modals.deleteSlide, $slidePanel);
        });
    }

    /**
     * Handles the create button click event.
     */
    function _onCreateButtonClick() {
        // Make a clone of the slide panel template and create a new element.
        var $slidePanel = $(elements.templates.slidePanel.clone().html());

        // Tab content element for selected language.
        var $tabBody = $this.find(selectors.tabBody).filter(':visible');

        // Slide panels.
        var $slidePanels = $tabBody.find(selectors.slidePanel);

        // Next panel index.
        var panelIndex = $slidePanels.length + 1;

        // Title for new slide panel.
        var newTitle = jse.core.lang.translate('NEW_SLIDE', 'sliders') + ' ' + panelIndex;

        // Add title to slide panel header.
        $slidePanel.find(selectors.slidePanelTitle).text(newTitle);

        // Add title to input field.
        $slidePanel.find(selectors.inputs.title).val(newTitle);

        // Add values to dropdowns.
        if ($slidePanels.length) {
            // Get all image dropdowns of the first panel.
            var $dropdowns = $slidePanels.first().find(selectors.inputs.dropdown);

            // Get the thumbnail dropdown options.
            var $thumbnailOptions = $dropdowns.filter(selectors.inputs.thumbnailImageDropdown).children().clone();

            // Get the image dropdown options.
            var $imageOptions = $dropdowns.not(selectors.inputs.thumbnailImageDropdown).first().children().clone();

            // Replace thumbnail options in new slide panel.
            $slidePanel.find(selectors.inputs.thumbnailImageDropdown).empty().append($thumbnailOptions).val('');

            // Replace image options in new slide panel.
            $slidePanel.find(selectors.inputs.dropdown).not(selectors.inputs.thumbnailImageDropdown).empty().append($imageOptions).val('');
        }

        // Add new slide panel element to tab body with fade effect.
        $slidePanel.hide().prependTo($tabBody).fadeIn();

        // Initialize widgets and extensions on the new slide panel element.
        gx.widgets.init($slidePanel);
        gx.extensions.init($slidePanel);

        // Trigger change to show the right action icons.
        $slidePanel.find('select').trigger('change');

        // Register change, to make prompt on page unload.
        _registerChange();

        // Toggle submit buttons.
        toggleSubmitButtons();
    }

    /**
     * Handles the confirmation button click event in the delete confirmation modal.
     *
     * @param {jQuery} $modal Delete confirmation modal element.
     * @param {jQuery} $slidePanel Slide panel element.
     */
    function _onDeleteConfirmationButtonClick($modal, $slidePanel) {
        // Hide modal.
        $modal.modal('hide');

        // Fade out slide panel element and then remove it.
        $slidePanel.fadeOut(400, function () {
            // Remove slide panel.
            $slidePanel.remove();

            // Toggle submit buttons.
            toggleSubmitButtons();
        });

        // Register change, to make prompt on page unload.
        _registerChange();
    }

    /**
     * Handles the key-up event on the slide title input field.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onTitleKeyup(event) {
        // Title input field.
        var $input = $(event.target);

        // Slide panel title element.
        var $title = $input.parents(selectors.slidePanel).find(selectors.slidePanelTitle);

        // Transfer input value to slide panel title.
        $title.text($input.val());
    }

    /**
     * Handles the mouse-enter event on a configuration row.
     *
     * @param {jQuery.Event} event Triggered event.
     */
    function _onConfigRowMouseEnter(event) {
        // Configuration row element.
        var $row = $(event.target);

        // Image map edit icon.
        var $imageMapIcon = $row.find(selectors.icons.imageMap);

        // Image map data container list element.
        var $list = $row.find(selectors.imageMapDataList);

        // Return immediately, if the image map edit icon does not exist.
        if (!$imageMapIcon.length || !$list.length) {
            return;
        }

        if ($list.children().length) {
            $imageMapIcon.removeClass('fa-external-link').addClass('fa-external-link-square');
        } else {
            $imageMapIcon.removeClass('fa-external-link-square').addClass('fa-external-link');
        }
    }

    /**
     * Handles the click event on the save button.
     */
    function _onSubmitSave() {
        $this.parents('form').trigger('submit');
    }

    /**
     * Handles the click event on the refresh list item in the submit button group.
     */
    function _onSubmitRefresh() {
        $this.parents('form').trigger('submit', { refresh: true });
    }

    /**
     * Adds an image to the image dropdowns.
     *
     * @param {String} fileName Name of the selected file.
     * @param {Boolean} [thumbnailImagesOnly = false] Apply on thumbnail image dropdowns only?
     */
    function _addImageToDropdowns(fileName) {
        var thumbnailImagesOnly = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        // Select specific dropdowns.
        var $dropdowns = $this.find(selectors.inputs.dropdown)[thumbnailImagesOnly ? 'filter' : 'not'](selectors.inputs.thumbnailImageDropdown);

        // Create new image option element.
        var $option = $('<option>', { value: fileName, text: fileName, class: classes.newImage });

        // Append new options to dropdowns.
        $dropdowns.append($option);
    }

    /**
     * Deletes an image from the image dropdowns.
     *
     * @param {String} fileName Name of the selected file.
     * @param {Boolean} [thumbnailImagesOnly = false] Apply on thumbnail image dropdowns only?
     */
    function _deleteImageFromDropdowns(fileName) {
        var thumbnailImagesOnly = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

        // Select all dropdowns.
        var $dropdowns = $this.find(selectors.inputs.dropdown)[thumbnailImagesOnly ? 'filter' : 'not'](selectors.inputs.thumbnailImageDropdown);

        // Remove image option from each dropdown.
        $dropdowns.each(function (index, element) {
            // Dropdown element.
            var $dropdown = $(element);

            // Remove option.
            $dropdown.find('[value="' + fileName + '"]').remove();

            // Set to default value if there are no image file options.
            if ($dropdown.children().length <= 1) {
                $dropdown.val('');
            }

            // Trigger change.
            $dropdown.trigger('change');
        });
    }

    /**
     * Disables or enables the submit buttons.
     */
    function toggleSubmitButtons() {
        // Enable the submit buttons?
        var doEnableSubmitButtons = true;

        // Slides.
        var $slides = $this.find(selectors.slidePanel);

        // Disable submit buttons, if there are no slides.
        if (!$slides.length) {
            doEnableSubmitButtons = false;
        }

        // Disable/Enable submit buttons.
        elements.buttons.submit.children().not('ul').prop('disabled', !doEnableSubmitButtons);
    }

    // --------------------------------------------------------------------
    // INITIALIZATION
    // --------------------------------------------------------------------

    module.init = function (done) {
        // Attach click event handler to sort button.
        elements.buttons.sort.on('click', _onSortButtonClick);

        // Attach event handlers to sort actions, slide panel delete button and inputs fields.
        $this.on('sortstart', _onSortStart).on('sortstop', _onSortStop).on('click', selectors.icons.delete, _onDeleteIconClick).on('keyup', selectors.inputs.title, _onTitleKeyup).on('change', selectors.inputs.file, _onImageAdd).on('click', selectors.icons.upload, _onUploadIconClick).on('click', selectors.icons.imageDelete, _onImageDelete).on('click', selectors.icons.imageMap, _onImageMap).on('change', selectors.inputs.dropdown, _onImageChange).on('mouseenter', selectors.configurationRow, _onConfigRowMouseEnter);

        // Attach event listeners to submit buttons.
        elements.buttons.submitSave.on('click', _onSubmitSave);
        elements.buttons.submitRefresh.on('click', _onSubmitRefresh);

        // Attach click event handler to create button.
        elements.buttons.create.on('click', _onCreateButtonClick);

        // Activate first tab.
        elements.tabHeader.children().first().addClass('active');

        // Activate first tab content.
        $this.find(selectors.tabBody).first().addClass('active in');

        // Trigger dropdown change event to hide the remove icon, if 'do not use' is selected.
        $this.find(selectors.inputs.dropdown).trigger('change', [false]);

        // Finish initialization.
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
